// -----------------------------------------------------------------------------
// Handler for the /authenticate endpoint. This endpoint validates a user's
// username and password against the DynamoDB users table. When successful, the
// service returns a lightweight token—simply a base64-encoded username.
//
// This token is not intended as a security boundary but serves as a stable
// identifier for follow-up authorization steps. All sensitive access control is
// delegated to Basic Auth and follow-up authorization endpoints.
// -----------------------------------------------------------------------------

import AWS from 'aws-sdk';
import { checkBasicAuth, generateToken } from './utils.js';

// DynamoDB DocumentClient is used for JSON-friendly reads from the "users" table.
const dynamodb = new AWS.DynamoDB.DocumentClient();

/**
 * Main entry point for POST /authenticate.
 *
 * Expected request payload:
 *   { "username": "...", "password": "..." }
 *
 * Authentication behaviour:
 * - Requires valid HTTP Basic Auth.
 * - Requires both username and password in the request body.
 * - Looks up the user in DynamoDB.
 * - If the user exists and the password matches, returns a token.
 * - If authentication fails, returns an empty token with HTTP 200.
 *
 * This endpoint intentionally normalizes failure responses to avoid exposing
 * sensitive details. The consumer can simply interpret an empty token as a
 * failed authentication.
 */
export async function handleAuthenticate(event) {
  // Gateway-level authentication guard.
  if (!checkBasicAuth(event.headers)) {
    return {
      statusCode: 403,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  try {
    // Parse request payload.
    const body = JSON.parse(event.body || '{}');
    const { username, password } = body;

    // Input validation: both fields must be provided.
    if (!username || !password) {
      return {
        statusCode: 422,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ token: '' })
      };
    }

    // Look up user record.
    const result = await dynamodb.get({
      TableName: 'users',
      Key: { username }
    }).promise();

    const user = result.Item;

    // Password mismatch or unknown user → return empty token.
    if (!user || user.password !== password) {
      return {
        statusCode: 200,
        body: JSON.stringify({ token: '' })
      };
    }

    // Success → return base64(username) as a persistent token.
    const token = generateToken(username);
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token })
    };

  } catch (err) {
    // Unexpected errors (e.g., DynamoDB connectivity) are surfaced as 500.
    return {
      statusCode: 500,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '', error: err.message })
    };
  }
}