// -----------------------------------------------------------------------------
// Handler for the /authenticate_via_ticket endpoint. This endpoint provides an
// alternative authentication mechanism using a temporary session ticket
// (typically issued by an external SSO system). The ticket contains a payload
// encoded in base64/base64url format, usually structured as:
//   { "user": "<username>", "time": <timestamp> }
//
// If the ticket is valid and resolves to a known user, the endpoint returns the
// same persistent token format as /authenticate: a base64-encoded username.
//
// Failure cases intentionally return HTTP 200 with an empty token to match the
// behaviour of the primary authentication endpoint.
// -----------------------------------------------------------------------------

import AWS from 'aws-sdk';
import { checkBasicAuth, parseRequestBody, generateToken } from './utils.js';

const dynamodb = new AWS.DynamoDB.DocumentClient();

/**
 * Decodes a string that may be formatted either as standard base64 or the
 * URL-safe base64url variant. Padding is added when necessary.
 */
function decodeBase64Flexible(input) {
  try {
    let normalized = input.replace(/-/g, '+').replace(/_/g, '/');
    const remainder = normalized.length % 4;

    if (remainder === 2) normalized += '==';
    else if (remainder === 3) normalized += '=';

    return Buffer.from(normalized, 'base64').toString('utf8');
  } catch {
    return null;
  }
}

/**
 * Main entry point for POST /authenticate_via_ticket.
 *
 * Behaviour mirrors /authenticate, but instead of username/password, the
 * client sends a ticket. After validating Basic Auth, the handler:
 * - Parses JSON payload { ticket }
 * - Decodes the ticket into a JSON structure
 * - Extracts the username
 * - Looks up that username in DynamoDB
 * - Returns a persistent token if the user exists
 */
export async function handleAuthenticateViaTicket(event) {
  if (!checkBasicAuth(event.headers)) {
    return {
      statusCode: 403,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  const input = parseRequestBody(event.body);
  if (!input || typeof input.ticket !== 'string' || input.ticket.trim() === '') {
    return {
      statusCode: 400,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  if (input.ticket.length > 256) {
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  const decoded = decodeBase64Flexible(input.ticket);
  if (!decoded) {
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  let payload;
  try {
    payload = JSON.parse(decoded);
  } catch {
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  const username = (payload?.user || '').toLowerCase().trim();
  if (!username) {
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '' })
    };
  }

  try {
    const result = await dynamodb.get({
      TableName: 'users',
      Key: { username }
    }).promise();

    if (!result.Item) {
      return {
        statusCode: 200,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ token: '' })
      };
    }

    const token = generateToken(username);
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token })
    };

  } catch (err) {
    return {
      statusCode: 500,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: '', error: err.message })
    };
  }
}
