// -----------------------------------------------------------------------------
// Handler for the core /authorize endpoint. This endpoint verifies whether a
// user is entitled to access protected content. Authorization is based on two
// factors:
//   1. A valid token that resolves to an existing user in DynamoDB.
//   2. A customer-defined external product authorization check.
//
// The logic is intentionally simple to allow integrators to replace or extend
// the external product ID rules. The endpoint returns a boolean `granted` flag
// wrapped in a 200 JSON response. Authorization failures never produce 4xx
// errors (except invalid Basic Auth), keeping the API contract predictable.
// -----------------------------------------------------------------------------

import {
  checkBasicAuth,
  parseRequestBody,
  validateRequestWithUser,
  isExternalProductIdAuthorized
} from './utils.js';

/**
 * Main entry point for POST /authorize.
 *
 * Flow overview:
 * - Reject the request if Basic Auth fails.
 * - Parse request JSON and fail with 400 for malformed payload.
 * - Validate the provided token and load the corresponding user.
 * - Apply the external product ID rule.
 * - Combine both results into a final `granted` decision.
 *
 * The API always returns a structured JSON response:
 *   { "granted": true | false }
 */
export async function handleAuthorize(event) {
  if (!checkBasicAuth(event.headers)) {
    return { statusCode: 403 };
  }

  const input = parseRequestBody(event.body);
  if (!input) {
    return { statusCode: 400 };
  }

  // The only mandatory field for this endpoint.
  const requiredFields = ['token'];

  // Validate token and load the user.
  const result = await validateRequestWithUser(input, requiredFields);

  // Apply external product ID check (integration-specific).
  const externalAuthorized = isExternalProductIdAuthorized(input);

  // Final decision: both user validation and external authorization must pass.
  const granted = result.granted && externalAuthorized;

  return {
    statusCode: 200,
    body: JSON.stringify({ granted })
  };
}