// -----------------------------------------------------------------------------
// Handler for the /authorize_article endpoint.
//
// This endpoint checks whether a user is entitled to view or access a specific
// article. The entitlement logic follows the same pattern used across all
// authorization endpoints in this service:
//
//   • Basic Auth protects the route.
//   • The incoming request body must contain a valid, decodable token.
//   • The token resolves to a user stored in DynamoDB.
//   • External product authorization rules are applied.
//
// Only when *both* user validation and external product authorization succeed
// does the endpoint return `granted: true`.
//
// The response always uses HTTP 200 with a JSON body such as:
//    { "granted": true | false }
//
// This predictable response format avoids exposing sensitive failure details to
// API consumers, while still providing clear access control semantics.
// -----------------------------------------------------------------------------

import {
  checkBasicAuth,
  parseRequestBody,
  validateRequestWithUser,
  isExternalProductIdAuthorized
} from './utils.js';

/**
 * Main entry point for POST /authorize_article.
 *
 * This handler ensures that the requester is authenticated, the token is valid,
 * the user exists, and they meet external entitlement rules. The external rules
 * (e.g., product-level permissions, content packs, entitlements) are delegated
 * to `isExternalProductIdAuthorized`, which can be replaced or extended by
 * integrators.
 */
export async function handleAuthorizeArticle(event) {
  // Gateway-level authentication: denies unauthorized clients immediately.
  if (!checkBasicAuth(event.headers)) {
    return { statusCode: 403 };
  }

  // Parse incoming JSON body safely.
  const input = parseRequestBody(event.body);
  if (!input) {
    return { statusCode: 400 };
  }

  // Token is mandatory for user context resolution.
  const requiredFields = ['token'];

  // Validate token, decode username, load user from DynamoDB.
  const result = await validateRequestWithUser(input, requiredFields);

  // Check whether this request is allowed based on external product logic.
  const externalAuthorized = isExternalProductIdAuthorized(input);

  // Final authorization result.
  const granted = result.granted === true && externalAuthorized === true;

  return {
    statusCode: 200,
    body: JSON.stringify({ granted })
  };
}