// -----------------------------------------------------------------------------
// Handler for the /authorize_chatbot endpoint.
//
// This endpoint determines whether a user is allowed to access chatbot-related
// features. The logic mirrors the behaviour of other entitlement endpoints:
//
//   • Basic Auth protects the route at the gateway level.
//   • The request body must contain a valid token.
//   • The token is decoded and validated against DynamoDB.
//   • External product rules are applied (integration-specific).
//
// Authorization is granted only when both the token/user validation and the
// external product authorization succeed.
//
// The handler always returns HTTP 200 with a `{ granted: boolean }` payload,
// unless Basic Auth or the request body is invalid.
// -----------------------------------------------------------------------------

import {
  checkBasicAuth,
  parseRequestBody,
  validateRequestWithUser,
  isExternalProductIdAuthorized
} from './utils.js';

/**
 * Main entry point for POST /authorize_chatbot.
 *
 * This method follows the same structure as /authorize_download and
 * /authorize_article to keep the entitlement system predictable and easy to
 * extend. The business logic for chatbot-specific authorization lives inside
 * `isExternalProductIdAuthorized` (integration-specific).
 */
export async function handleAuthorizeChatbot(event) {
  // Protect the endpoint with HTTP Basic Auth.
  if (!checkBasicAuth(event.headers)) {
    return { statusCode: 403 };
  }

  // Parse incoming request JSON.
  const input = parseRequestBody(event.body);
  if (!input) {
    return { statusCode: 400 };
  }

  // Only the token field is mandatory for chatbot authorization.
  const requiredFields = ['token'];

  // Validate token presence, decode token, and load user from DynamoDB.
  const result = await validateRequestWithUser(input, requiredFields);

  // External authorization rule: determines whether the chatbot feature
  // is allowed for this user/product setup.
  const externalAuthorized = isExternalProductIdAuthorized(input);

  // Final access decision.
  const granted = result.granted && externalAuthorized;

  return {
    statusCode: 200,
    body: JSON.stringify({ granted })
  };
}