// -----------------------------------------------------------------------------
// Handler for the /authorize_download endpoint.
//
// This endpoint determines whether a user is allowed to download protected
// content. Authorization depends on two factors:
//   • A valid token that resolves to a real user in DynamoDB
//   • External product validation, defined by the integration environment
//
// Both conditions must evaluate to true for the request to be granted.
//
// Clients always receive a JSON response with a `granted` boolean. Forbidden
// Basic Auth attempts return HTTP 403. Malformed request bodies return 400.
// -----------------------------------------------------------------------------

import {
  checkBasicAuth,
  parseRequestBody,
  validateRequestWithUser,
  isExternalProductIdAuthorized
} from './utils.js';

/**
 * Main entry point for POST /authorize_download.
 *
 * This handler performs gateway-level Basic Auth validation first. Afterward it
 * parses the JSON body, verifies the presence of a valid token, loads the user
 * from DynamoDB, and then applies the external product authorization rule.
 *
 * The final `granted` decision is the logical AND of both checks.
 */
export async function handleAuthorizeDownload(event) {
  // HTTP Basic Auth protects this endpoint at the gateway layer.
  if (!checkBasicAuth(event.headers)) {
    return {
      statusCode: 403,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ granted: false, error: 'Forbidden' })
    };
  }

  // Parse JSON body safely.
  const input = parseRequestBody(event.body);
  if (!input) {
    return { statusCode: 400 };
  }

  // The token is always required.
  const requiredFields = ['token'];

  // Validate the token and load the corresponding user.
  const result = await validateRequestWithUser(input, requiredFields);

  // External product validation: integrator-specific rules.
  const externalAuthorized = isExternalProductIdAuthorized(input);

  // Both conditions must be true for access to be permitted.
  const granted = result.granted && externalAuthorized;

  return {
    statusCode: 200,
    body: JSON.stringify({ granted })
  };
}