// -----------------------------------------------------------------------------
// Central Lambda router. All HTTP POST requests for this service arrive here
// via API Gateway. The handler inspects the request path and forwards the
// request to the appropriate endpoint implementation.
//
// This router intentionally does not implement complex route matching logic.
// Instead, it relies on simple string checks (path.endsWith(...)) to match the
// defined API paths. This keeps the routing layer predictable and easy to
// maintain while avoiding unnecessary dependencies.
//
// Each endpoint returns a unified response object containing `statusCode` and an
// optional JSON-encoded `body`. The router normalizes the response: successful
// endpoint handlers always return JSON with proper headers.
// -----------------------------------------------------------------------------

import { handleAuthenticate } from './authenticate.js';
import { handleAuthenticateViaTicket } from './authenticate_via_ticket.js';
import { handleAuthorize } from './authorize.js';
import { handleAuthorizeDownload } from './authorize_download.js';
import { handleAuthorizeArticle } from './authorize_article.js';
import { handleAuthorizeChatbot } from './authorize_chatbot.js';
import { handleIssues } from './issues.js';

/**
 * AWS Lambda entrypoint. Processes an API Gateway event and routes POST
 * requests to the appropriate handler.
 *
 * The router makes no assumptions about the structure of the request beyond
 * the fields provided by API Gateway v1/v2. Both `rawPath` and `path` are
 * supported, as are both `requestContext.http.method` and `httpMethod`.
 */
export async function handler(event) {
  const path = event.rawPath || event.path || '';
  const method = event.requestContext?.http?.method || event.httpMethod;

  console.log('[index] Received input:', event.body);

  // Only POST endpoints are defined for this API.
  if (method === 'POST') {
    let response;

    // Endpoint dispatching. Each check is isolated so new endpoints can be
    // added with minimal friction.
    if (path.endsWith('/authenticate')) {
      response = await handleAuthenticate(event);
    }

    if (path.endsWith('/authenticate_via_ticket')) {
      response = await handleAuthenticateViaTicket(event);
    }

    if (path.endsWith('/authorize_download')) {
      response = await handleAuthorizeDownload(event);
    }

    if (path.endsWith('/authorize_article')) {
      response = await handleAuthorizeArticle(event);
    }

    if (path.endsWith('/authorize_chatbot')) {
      response = await handleAuthorizeChatbot(event);
    }

    if (path.endsWith('/authorize')) {
      response = await handleAuthorize(event);
    }

    if (path.endsWith('/issues')) {
      response = await handleIssues(event);
    }

    // When a handler returns a valid response object, normalize and return it.
    // Handlers are expected to always return a `statusCode`.
    if (response?.statusCode !== undefined) {
      console.log('[response]', response);
      return {
        statusCode: response.statusCode,
        headers: { 'Content-Type': 'application/json' },
        body: response.body,
      };
    }
  }

  // Non-POST requests or unknown paths fall through to a minimal 404 response.
  return {
    statusCode: 404,
    body: JSON.stringify({ message: 'Not found', path })
  };
}
