// -----------------------------------------------------------------------------
// Handler for the /issues endpoint. This endpoint returns a list of accessible
// magazine issues for the authenticated user. Access requires valid Basic Auth
// and a valid token that maps to an existing user record in DynamoDB.
//
// The endpoint intentionally behaves permissively on authorization failures:
// when token validation fails, it returns an empty issues array with HTTP 200.
// This mirrors the behaviour of the original API design and maintains backward
// compatibility for consuming clients.
// -----------------------------------------------------------------------------

import { checkBasicAuth, parseRequestBody, validateRequestWithUser } from './utils.js';

/**
 * Main entry point for POST /issues.
 *
 * Flow:
 * - Reject immediately if Basic Auth is missing or invalid.
 * - Parse the request body into JSON. If parsing fails, return 400.
 * - Validate the token and ensure the user exists.
 * - On validation failure, return an empty issues array.
 * - On success, return a fixed set of sample issues.
 *
 * This handler can later be extended to return issue lists based on user
 * permissions, categories, or subscriptions once the business rules mature.
 */
export async function handleIssues(event) {
  if (!checkBasicAuth(event.headers)) {
    return {
      statusCode: 403
    };
  }

  const input = parseRequestBody(event.body);
  if (!input) {
    return {
      statusCode: 400
    };
  }

  // Token is the only required field for /issues.
  const requiredFields = ['token'];
  const result = await validateRequestWithUser(input, requiredFields);

  // API contract: when `granted` is false, return an empty list instead of an
  // authorization error. This prevents errors from propagating to clients and
  // allows them to treat the response uniformly.
  if (!result.granted) {
    return {
      statusCode: 200,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ issues: [] })
    };
  }

  // Successful authorization → return sample issues.
  // This placeholder content is used for testing until a production-grade
  // subscription or entitlement model is implemented.
  return {
    statusCode: 200,
    body: JSON.stringify({
      issues: [
        'external_id_2024_01', // external ID support
        'apple_product_identifier_2024_02', // Apple product ID support
        '123_google_product_id_2024_03' // Google product ID support
      ]
    })
  };
}
