# Auth API – Documentation

This service provides a lightweight authentication and authorization API designed for magazine/article publishing platforms. It is built for AWS Lambda + API Gateway and uses DynamoDB as the single source of truth for user data.

The goal is to keep the system simple, predictable, and easily extendable by integrators.

---

## Overview

This API includes endpoints for:

* **/authenticate** – Validate username + password and return a persistent token.
* **/authenticate_via_ticket** – Authenticate using a base64/base64url-encoded SSO ticket.
* **/authorize** – Check general access based on token + external product ID rules.
* **/authorize_download** – Check download entitlement.
* **/authorize_article** – Check article access entitlement.
* **/authorize_chatbot** – Check chatbot feature entitlement.
* **/issues** – Return user-specific magazine issues.

Each endpoint follows the same architectural patterns:

* Basic Auth at gateway level
* JSON request body validation
* Token decoding → user lookup in DynamoDB
* Optional external product authorization
* Always return 200 with predictable JSON body (except Basic Auth or invalid JSON)

---

## Tech Stack

| Component   | Description                              |
| ----------- | ---------------------------------------- |
| Runtime     | AWS Lambda (Node.js, ES Modules)         |
| Language    | JavaScript (ESM)                         |
| Persistence | DynamoDB (users table)                   |
| Auth Layer  | Basic Auth + Token-based user resolution |
| Packaging   | Zip bundle via `npm run build`           |

---

## Project Structure

```
.
├── index.js                      # Main router for all POST endpoints
├── utils.js                      # Shared logic (Basic Auth, tokens, DynamoDB lookup)
├── authenticate.js               # Username/password authentication
├── authenticate_via_ticket.js    # Ticket-based SSO authentication
├── authorize.js                  # Generic entitlement check
├── authorize_download.js         # Download entitlement
├── authorize_article.js          # Article entitlement
├── authorize_chatbot.js          # Chatbot entitlement
├── issues.js                     # Provides list of accessible issues
├── package.json
└── auth_openapi.yaml             # API specification (used for reference)
```

---

## Token Model

Tokens are intentionally simple: the username encoded in Base64.

Example:

```
"john@example.com" → "am9obkBleGFtcGxlLmNvbQ=="
```

These tokens are **not** security tokens. They are identifiers. All security is handled by:

* Basic Auth (gateway-level)
* DynamoDB-based authorization

---

## Endpoint Flow Summary

### 1. Basic Auth Validation

Every endpoint except `/authenticate*` enforces a strict Basic Auth check:

```
Authorization: Basic base64(username:password)
```

If invalid → `403 Forbidden`.

---

### 2. JSON Parsing

`parseRequestBody()` ensures malformed JSON never breaks the system.

If invalid JSON → `400`.

---

### 3. Token Validation

`validateRequestWithUser()` ensures:

* The token exists
* The token decodes into a username
* A user with that username exists in DynamoDB
* All required fields are present

If any checks fail, authorization endpoints simply return:

```
200 { "granted": false }
```

---

### 4. External Product Authorization

This function:

```
isExternalProductIdAuthorized(input)
```

Is **customer-specific**. It determines whether the incoming request should be authorized beyond just the token.

Currently placeholder logic:

```
input.product_id_external === "authorized"
```

---

### 5. Final Authorization Logic

All entitlement endpoints compute:

```
granted = result.granted && externalAuthorized
```

Then respond with:

```
200 { "granted": true/false }
```

---

## DynamoDB Schema

Table **users**:

* `username` (PK, string)
* `password` (string)
* Additional entitlement metadata as needed

Example item:

```
{
  "username": "john@example.com",
  "password": "123456",
  "permissions": ["issue", "download"],
  "issue_name": "sample_issue_2020_01",
  "category_ids": "news,tech,science",
  "product_id_external": "authorized"
}
```

---

## Local Development

The project uses a simple build script to create a Lambda-compatible zip bundle.

### Installation

```
npm install
```

### Build Deployment Package

```
npm run build
```

Output:

```
function.zip
```

This zip file can be uploaded to AWS Lambda.

---

## Error Handling Philosophy

* Invalid Basic Auth → **403**
* Invalid JSON → **400**
* DynamoDB failure → **500** (only during authenticate/authenticate_via_ticket)
* All other failures → **200 with empty token or granted: false**

This leads to a predictable, stable contract across all clients.

---

## Extending the API

You can add new authorization endpoints extremely easily:

1. Create `authorize_newfeature.js`
2. Copy the control flow from an existing file
3. Add your custom required fields or external logic
4. Add a new `if (path.endsWith('/authorize_newfeature'))` block in `index.js`

---

## Deployment Notes

* This Lambda uses native ESM (`type: "module"` in package.json)
* AWS SDK v2 is bundled via dependencies
* No bundler required—zip and deploy

---

## License

Internal / Proprietary — Not meant for public distribution.

---

If you need a sequence diagram, architecture sketch, or onboarding doc for new developers, just tell me!
