// -----------------------------------------------------------------------------
// Shared utility functions used across the authentication and authorization API.
// Provides JSON parsing, HTTP Basic Auth validation, token encoding/decoding,
// and a unified DynamoDB-backed user lookup with field validation.
// -----------------------------------------------------------------------------

import AWS from 'aws-sdk';

// DocumentClient wrapper for reading JSON-friendly objects from DynamoDB.
const dynamodb = new AWS.DynamoDB.DocumentClient();

/**
 * Attempts to parse a JSON request body into an object. Returns `null` on any
 * parsing failure. This avoids throwing inside Lambda handlers and ensures
 * consistent behaviour across all endpoints.
 */
export function parseRequestBody(body) {
  try {
    return JSON.parse(body || '{}');
  } catch (err) {
    console.error('[parseRequestBody] Invalid JSON:', err.message);
    return null;
  }
}

/**
 * Validates the presence and correctness of an HTTP Basic Authorization header.
 *
 * This implementation uses fixed credentials for gateway-level authentication.
 * The header must follow the pattern: `Authorization: Basic <base64(user:pass)>`.
 *
 * Returns `true` when the credentials match, otherwise `false`.
 */
export function checkBasicAuth(headers) {
  const VALID_USER = 'pressmatrix';
  const VALID_PASS = "we'rereallysecure!";

  const authHeader = headers?.authorization || '';
  if (!authHeader.startsWith('Basic ')) {
    return false;
  }

  try {
    const encoded = authHeader.slice(6); // "Basic " prefix length
    const decoded = Buffer.from(encoded, 'base64').toString();
    const [user, pass] = decoded.split(':');
    return user === VALID_USER && pass === VALID_PASS;
  } catch {
    return false;
  }
}

/**
 * Encodes a username into a base64 token. Tokens are intentionally simple:
 * they directly map to the username and are not cryptographically secure.
 */
export function generateToken(username) {
  return Buffer.from(username).toString('base64');
}

/**
 * Decodes a base64 token back into a username. If decoding fails or yields an
 * empty result, `null` is returned.
 */
export function decodeToken(token) {
  try {
    const decoded = Buffer.from(token, 'base64').toString();
    return decoded?.trim() || null;
  } catch {
    return null;
  }
}

/**
 * Loads a user from DynamoDB and validates that all required request fields
 * exist. Field values may be empty; the API only enforces presence.
 *
 * The returned object follows a unified contract used by all authorization
 * endpoints: `{ granted: boolean, user?, error?, missingField? }`.
 */
export async function validateRequestWithUser(input, requiredFields = []) {
  if (!input.token || input.token.trim() === "") {
    return { granted: false, error: 'Missing token' };
  }

  const username = decodeToken(input.token);
  if (!username) {
    return { granted: false, error: 'Invalid token format' };
  }

  let user;
  try {
    const result = await dynamodb.get({
      TableName: 'users',
      Key: { username },
    }).promise();

    user = result.Item;
    if (!user) {
      return { granted: false, error: 'User not found' };
    }
  } catch (err) {
    console.error('[validateRequestWithUser] DynamoDB error:', err);
    return { granted: false, error: 'DynamoDB error' };
  }

  for (const field of requiredFields) {
    if (!(field in input)) {
      return { granted: false, error: `Missing required: ${field}`, missingField: field };
    }
  }

  return { granted: true, user };
}

/**
 * Evaluates whether the provided external product identifier is considered
 * authorized. This function currently contains placeholder logic and is meant
 * to be replaced by customer-specific validation rules.
 */
export function isExternalProductIdAuthorized(input) {
  console.log(input);

  if (!input.product_id_external) {
    return false;
  }

  // Placeholder demo rule: only the literal value "authorized" passes.
  if (input.product_id_external === 'authorized') {
    return true;
  }

  return false;
}
